/*
 * Shue.cpp
 *
 *  Created on: Aug 11, 2008
 *      Author: dberrios
 */

#include "Shue.h"
#include <algorithm>
#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>
#include <cmath>
#include <cstdlib>

using namespace std;

/**
 *
 */
IMF::IMF()
{
	// TODO Auto-generated constructor stub
	//setenv("TZ", "UTC", 1);

}

//need to pass low = 0 and high = n-1 initially
/**
 * @param targetTime
 * @param low
 * @param high
 * @return
 */
int IMF::bSearchTime(const time_t& targetTime, int low, int high)
{
	//cout << "low: " << low << " high: " << high << endl;
	//cout << "Inside bSearch.  targetTime: " << asctime(gmtime(&targetTime));
	if (low > high)
		return -1;
	int mid = (low + high) / 2;

	int result = betweenDateComparison(targetTime, mid);
	//cout << "result: " << result << endl;
	if (result > 0)
	{
		return bSearchTime(targetTime, mid+1, high);

	} else if (result < 0)
	{
		return bSearchTime(targetTime, low, mid-1);
	} else
	{
		return mid;
	}

}

/**
 * @param targetTime
 * @param low
 * @param high
 * @return
 */
float IMF::bSearchMagnetopauseStart(const time_t& targetTime, const float& low, const float& high)
{
	float mid = (low + high)/2.0f;
	if (high - low < 1.e-4)
	{
		return mid;
	}
	else
	{
		int value = magnetopauseComparison(targetTime, mid);
		if (value < 0)
		{
			return bSearchMagnetopauseStart(targetTime, low, mid);
		} else
		{
			return bSearchMagnetopauseStart(targetTime, mid, high);
		}

	}

}

/**
 * returns -1 or 1 depending on whether the x position is inside the magnetopause or not
 * the binary search will determine when to stop, based on some minimum delta between
 * the low and high x position
 */
/**
 * @param targetTime
 * @param mid
 * @return
 */
int IMF::magnetopauseComparison( const time_t& targetTime, const float& mid)
{
	bool isInside;
	float radius;
	interpolate(targetTime, mid, radius, isInside);
	if (isInside)
		return 1;
	else
		return -1;
	//time_t targetTime, float xPosition, float& radius, bool& isInside)
}

/**
 * @param targetTime
 * @param mid
 * @return
 */
int IMF::exactDateComparison ( const time_t& targetTime, int mid)
{
	if ( targetTime > imfdata[mid].epoch)
	{
		return 1;
	} else if (targetTime < imfdata[mid].epoch)
	{
		return -1;
	}
	else
	{
		return 0;
	}
}


//because of some weird differences in calculating the epochs, we'll just compare the strings
//generated by asctime
/**
 * @param targetTime
 * @param test
 * @return
 */
int IMF::betweenDateComparison (const time_t& targetTime, int test)
{
	string targetTimeString = asctime(gmtime(&targetTime));
	string lowString = asctime(gmtime(&imfdata[test].epoch));
	string highString = asctime(gmtime(&imfdata[test+1].epoch));
	//cout << "Inside betweenDateComparison.  targetTime: " << targetTimeString << " " << lowString << " " << highString << endl;
	if (targetTimeString < lowString && targetTimeString < highString)
	{
		return -1;
	} else if (targetTimeString >= lowString && targetTimeString <= highString)
	{
		return 0;
	} else
		return 1;
}

/**
 * @param filename
 */
void IMF::open(string filename)
{
	//first, grab all lines in the file
	vector < IMFdata > lines;
	ifstream file;
	file.open(filename.c_str());
	if (file.fail())
	{
		cerr << "File " << filename << " could not be opened." << endl;
		exit(1);
	}
	string line;
	//cout << "before getline" << endl;
	bool start = false;
	getline(file, line);
	while (!file.eof())
	{
		//cout << line << endl;
		if (start == false && (line.find("#START") != string::npos))
		{
			//beginning of data found.  skip the
			//line with #START
			start = true;
			getline(file, line);
		}
		if (start == true)
		{
			//cout << " processing line. " << endl;
			IMFdata currentData = parse(line);
			//cout << asctime(&currentData.time);
			imfdata.push_back(currentData);

		} else
		{
			//cout << " not processing line. " << endl;
		}
		getline(file, line);



	}
	//cout << "after getline" << endl;
//cout << line.npos << endl;


}

/**
 * @param line
 * @return
 */
IMFdata IMF::parse(const string &line)
{
	// Skip delimiters at beginning.
	string delimiters = "\n \t";
	vector<string> tokens;
	IMFdata imfdataline;

	//find the position of the character that does not match any of the delimiters
	//cout << "here" << endl;
	if (line.length() > 0)
	{
		size_t lastPosition = line.find_first_not_of(delimiters, 0);
		// Find first delimiter after the lastPosition
		size_t currentPosition = line.find_first_of(delimiters, lastPosition);

		while (currentPosition != string::npos || lastPosition != string::npos)
		{
			tokens.push_back(line.substr(lastPosition, currentPosition - lastPosition));

			// Find next character position that does not match any of the delimiters
			lastPosition = line.find_first_not_of(delimiters, currentPosition);

			// Find next delimiter
			currentPosition = line.find_first_of(delimiters, lastPosition);
		}

	}

	//tm year is the number of years from 1900
	imfdataline.year = parseInt(tokens[0]) - 1900;
	imfdataline.month = parseInt(tokens[1]);
	imfdataline.day = parseInt(tokens[2]);
	imfdataline.hour = parseInt(tokens[3]);
	imfdataline.min = parseInt(tokens[4]);
	imfdataline.sec = parseInt(tokens[5]);
	imfdataline.msec = parseInt(tokens[6]);
	imfdataline.bx = parseFloat(tokens[7]);
	imfdataline.by = parseFloat(tokens[8]);
	imfdataline.bz = parseFloat(tokens[9]);
	imfdataline.vx = parseFloat(tokens[10]);
	imfdataline.vy = parseFloat(tokens[11]);
	imfdataline.vz = parseFloat(tokens[12]);
	imfdataline.n = parseFloat(tokens[13]);
	imfdataline.t = parseFloat(tokens[14]);
	tm time_tm;
	time_tm.tm_year = imfdataline.year;
	time_tm.tm_mon = imfdataline.month;
	time_tm.tm_mday = imfdataline.day;
	time_tm.tm_hour = imfdataline.hour;
	time_tm.tm_min = imfdataline.min;
	time_tm.tm_sec = imfdataline.sec;
	//time_tm.tm_zone = "GMT";

	imfdataline.epoch = mktime(&time_tm);

	float n_to_rho = 1.602e-21;
	imfdataline.d_p = n_to_rho*imfdataline.n/2.*imfdataline.vx*imfdataline.vx * 1.e15;

	if (imfdataline.bz >= 0.)
	{
		imfdataline.shue_r0 = (11.4+0.013*imfdataline.bz)*pow(imfdataline.d_p, (-1.f/6.6f));
	} else
	{
		imfdataline.shue_r0 = (11.4+0.140*imfdataline.bz)*pow(imfdataline.d_p, (-1.f/6.6f));
	}
	return imfdataline;



}


/**
 * @param str
 * @return
 */
int IMF::parseInt(const string &str)
{
	istringstream iss(str);
	int test;
	iss >> test;
	return test;
}

/**
 * @param str
 * @return
 */
float IMF::parseFloat(const string &str)
{
	istringstream iss(str);
	float test;
	iss >> test;
	return test;
}

/**
 * @param index
 * @param transitTime
 * @return
 */
IMFdata IMF::interpolateIMFData(int & index, time_t & transitTime)
{
    IMFdata before = imfdata[index];
    IMFdata after = imfdata[index + 1];
    float difference = transitTime - before.epoch;

    float range = after.epoch - before.epoch;
    float percentage = difference / range;
    //cout << "percentage: " << percentage << endl;
    if (percentage > 1.0)
    {
    	cerr << "percentage is greater than 1.0.  This is incorrect!!" << endl;
    }
    IMFdata newdataline;
    newdataline.epoch = transitTime;
    //newdataline.time = *gmtime(&transitTime);


    newdataline.bx = interpolateFloat(before.bx, after.bx, percentage);
    newdataline.by = interpolateFloat(before.by, after.by, percentage);
    newdataline.bz = interpolateFloat(before.bz, after.bz, percentage);
    newdataline.vx = interpolateFloat(before.vx, after.vx, percentage);
    newdataline.vy = interpolateFloat(before.vy, after.vy, percentage);
    newdataline.vz = interpolateFloat(before.vz, after.vz, percentage);
    newdataline.n = interpolateFloat(before.n, after.n, percentage);
    newdataline.t = interpolateFloat(before.t, after.t, percentage);
    newdataline.d_p = interpolateFloat(before.d_p, after.d_p, percentage);
    newdataline.shue_r0 = interpolateFloat(before.shue_r0, after.shue_r0, percentage);

    return newdataline;
}

/**
 * @param targetTime
 * @param xPositions
 * @param radii
 * @param nightSideExtent
 * @param slices
 */
void IMF::calculateMagnetopause(const time_t& targetTime, vector<float>& xPositions, vector<float>& radii, float nightSideExtent, int slices)
{
	//start at X= 0RE and perform a binary search towards the sun.
	//once the start of the magnetopause is found, calculate the radius
	//of the magnetopause at regular intervals from the start to the end of
	//the simulation.

	float radius;
	bool isInside;
	interpolate(targetTime, 0.f, radius, isInside );


	if (isInside) //start is good
	{
		float start = bSearchMagnetopauseStart(targetTime, 0.f, 33.f);
		float delta = (start - nightSideExtent)/(float)(slices-1);
		cout << "start of magnetopause: " << bSearchMagnetopauseStart(targetTime, 0.f, 33.f) << endl;
		radius;
		isInside;
		xPositions.push_back(start);
		radii.push_back(0.f);
		for (int i = 0; i < slices; i++)
		{
			float newstart = start - (float)i*delta;
			xPositions.push_back(newstart);
			interpolate(targetTime, newstart, radius, isInside);
			radii.push_back(radius);
		}

	} else
	{
		cerr << "crapping out.  0 RE was not inside the magnetopause!" << endl;
		exit(1);
	}

	for (int i = 0; i < xPositions.size(); i++)
	{
		cout << "x: " << xPositions[i] << " radius: " << radii[i] << endl;
	}


}

/**
 * Interpolates the IMF data and calculates the radius of the magnetopause at xPosition
 * and also determines whether the xPosition is inside or outside of the magnetopause
 */
/**
 * @param targetTime
 * @param xPosition
 * @param radius
 * @param isInside
 */
void IMF::interpolate(time_t targetTime, float xPosition, float& radius, bool& isInside)
{
	//this assumes the global magnetosphere run has the upstream boundary at 33 RE
	float xr_max = 33.;
	time_t targetTimeEpoch = targetTime;

	int index = bSearchTime(targetTimeEpoch, 0,imfdata.size()-2);

	if (index == -1)
	{
		cerr << "could not find targetTime" << endl;
		exit(1);
	}
	IMFdata dataline;
	if (imfdata[index].epoch == targetTimeEpoch)
	{
		dataline = imfdata[index];
	} else
	{
		dataline = interpolateIMFData(index, targetTimeEpoch);
	}
	time_t transitTime = targetTimeEpoch - (long)(6371.2*(xPosition - xr_max)/dataline.vx);

	index = bSearchTime(transitTime, 0, imfdata.size()-2);
	IMFdata newdataline;
	if (imfdata[index].epoch == transitTime)
	{
		newdataline = imfdata[index];
	} else
	{
	    newdataline = interpolateIMFData(index, transitTime);



	}
	float sw_bz = newdataline.bz;

	float sw_v_x = newdataline.vx;
	float sw_v_y = newdataline.vy;
	float sw_v_z = newdataline.vz;
	float sw_n = newdataline.n;
	float sw_dp = newdataline.d_p;

	time_t minTime = std::max(transitTime, imfdata[0].epoch);
	time_t tt = std::min(minTime, imfdata[imfdata.size()-1].epoch);

	index = bSearchTime(tt, 0, imfdata.size()-2);
	IMFdata newdataline2;
	if (imfdata[index].epoch == tt)
	{
		newdataline2 = imfdata[index];
	} else
	{
		newdataline2 = interpolateIMFData(index, tt);

	}
	float r1 = newdataline2.shue_r0;
	float shue_alpha = (0.58-0.01*sw_bz)*(1.+.01*sw_dp);
	float theta2 = atan2((double)15.,(double)xPosition);
	float theta = 0.;
	int count = 0;
	float shue_rr;
	while (abs(theta-theta2) > 1.e-4)
	{
		theta = theta2;
		shue_rr = r1 * pow(2.f/(1.f + cos(theta)),shue_alpha);
		theta2 = atan2((double)(abs(sin(theta))*shue_rr),(double)xPosition );
		count ++;
		if (count > 50)
			theta = theta2;

	}

	/*cout << "Angle from x-axis after iteration " << count << ": " <<  (theta*180.f/M_PI) << endl;

	cout << "*****************" << endl;
	cout << "Results of Shue()" << endl;
	cout << " X: " << xPosition << endl;
	cout << "X position (Shue_X): " << cos(theta)*shue_rr << endl;
	cout << "Distance of Shue magnetopause from Earth (Shue_RR): " << shue_rr << endl;
	cout << "Radius of Shue magnetopause (Shue_R) at position X: " << (sin(theta)*shue_rr) << endl;
	cout << "Inside magnetopause: " << (shue_rr > abs(xPosition)) << endl;
	cout << "Solar wind parameters used: Bz: " << sw_bz << " N: " << sw_n << endl;
	cout << "\tV: " << sw_v_x << " " << sw_v_y << " " << sw_v_z << "  P: " << sw_dp << " alpha: " << shue_alpha << endl;
	*/
	radius = (sin(theta)*shue_rr);
	isInside = (shue_rr > abs(xPosition));
}

/**
 * @param before
 * @param after
 * @param percentage
 * @return
 */
float IMF::interpolateFloat(float before, float after, float percentage)
{
	float range = after - before;
	return before + range*percentage;
}

/**
 * @param i
 * @return
 */
IMFdata& IMF::operator[] (int i)
{
	return imfdata[i];
}

/**
 *
 */
IMF::~IMF()
{
	// TODO Auto-generated destructor stub
}
